import pandas as pd

from .technical_analysis import TechnicalAnalysis

# Direction constants
BEARISH_CROSS, BULLISH_CROSS = (0, 1)


class CrossingEMAs(TechnicalAnalysis):
    """
    The bot implements a simple crossing moving average strategy using
    the 5 period EMA as the signal and the 20 period EMA.

    When a crossing happens the bot fills its position to the position limit
    in the respective direction and waits for a signal in the other direction.

    When this happens it unloads its positions and loads in the new direction
    using the same position limit as target.

    In case we get the same values for the two EMAs from the initial price curve,
    the bot will sell/buy when the signal drops below/rises above the base EMA.

    Inherits from:
        TechnicalAnalysis (class): The base class for technical analysis-based trading strategies.
    
    Attributes:
        small_ema_period (int): The period for the short/signal EMA.
        big_ema_period (int): The period for the long/base EMA.
        current_amount (int): The current position amount.
        small_ema (float): The latest value of the short/signal EMA.
        big_ema (float): The latest value of the long/base EMA.
        last_cross (int): The direction of the last EMA cross, either BEARISH_CROSS or BULLISH_CROSS.

    BEARISH_CROSS, BULLISH_CROSS = (0, 1)
    """

    def __init__(self, api, options):
        """
        Initializes the CrossingEMAs strategy with specific options for the EMA periods.

        Parameters:
            api (object): The API interface for interacting with the Electronic Broker (EB).
            options (dict): A dictionary containing 'small_ema_period' and 'big_ema_period' keys for the EMA periods.

        Raises:
            Exception: If the required keys are missing in `options`, or if the periods are not positive integers.
        """
        super().__init__(api, options)

        if not {'small_ema_period', 'big_ema_period'} <= set(options):
            raise Exception(
                f'CrossingEMAs options should contain the keys: small_ema_period, big_ema_period')

        self.small_ema_period = options['small_ema_period']
        self.big_ema_period = options['big_ema_period']

        options_list = [self.small_ema_period, self.big_ema_period]

        if not all(isinstance(x, int) and x > 0 for x in options_list):
            raise Exception('Periods must be positive integers')

        self.current_amount = 0
        self.small_ema = None
        self.big_ema = None
        self.last_cross = None
        self.compute_emas()

    def on_price_curve_change(self, price_curve):
        """
        Updates the strategy's internal state in response to a change in the price curve. Recomputes the EMAs based 
        on the new price curve.

        Parameters:
            new_price_curve (list): The updated price curve.
        """
        super().on_price_curve_change(price_curve)
        self.compute_emas()

    def compute_emas(self):
        """
        Computes the current values of the short/signal and long/base EMAs based on the latest price curve.
        """
        small_ema_series = pd.Series(self.price_curve).ewm(
            span=self.small_ema_period).mean()
        big_ema_series = pd.Series(self.price_curve).ewm(
            span=self.big_ema_period).mean()
        self.small_ema = round(small_ema_series.tolist()
                               [-1], self.api.price_decimals)
        self.big_ema = round(big_ema_series.tolist()
                             [-1], self.api.price_decimals)

    def run(self):
        """
        Executes the strategy's trading logic based on the current state of the EMAs. Identifies new EMA crosses 
        and adjusts the trading position accordingly.
        """
        cross_direction = self.get_new_cross_direction()

        if cross_direction is not None:
            self.last_cross = cross_direction
            self.load(cross_direction)

    def get_new_cross_direction(self):
        """
        Determines the direction of any new EMA cross based on the current EMA values.

        Returns:
            int: The direction of the cross (BEARISH_CROSS or BULLISH_CROSS), or None if no new cross has occurred.
        """
        cross_direction = None
        if self.last_cross in [BEARISH_CROSS, None] and self.small_ema > self.big_ema:
            cross_direction = BULLISH_CROSS
        elif self.last_cross in [BULLISH_CROSS, None] and self.small_ema < self.big_ema:
            cross_direction = BEARISH_CROSS
        return cross_direction

    def load(self, direction):
        """
        Adjusts the trading position based on the direction of the EMA cross, aiming to reach the target position 
        amount.

        Parameters:
            direction (int): The direction to load the position, either BEARISH_CROSS or BULLISH_CROSS.
        """
        sgn = (-1, 1)[direction == BULLISH_CROSS]
        remaining_amount = self.target_amount - sgn * self.current_amount
        if remaining_amount < 0:
            return
        else:
            while remaining_amount > 0 and direction == self.last_cross:
                ticket = self.api.max_ticket if remaining_amount >= self.api.max_ticket else remaining_amount
                amount = (self.api.hit_price(direction, ticket,
                          self.depth[direction][0]['price']))['amount']
                self.current_amount += sgn * amount
                remaining_amount -= amount

    def log_info(self):
        """
        Generates a log message describing the current state of the strategy, including the current amount, EMA values, 
        and the direction of the last cross.

        Returns:
            str: A log message summarizing the strategy's current state.
        """
        last_cross = "Bearish" if self.last_cross == BEARISH_CROSS \
            else "Bullish" if self.last_cross == BULLISH_CROSS else None

        return f'EMA     -- Amount: {self.current_amount}, {self.small_ema_period} Ema: {self.small_ema}, ' \
               f'{self.big_ema_period} Ema: {self.big_ema}, Last cross: {last_cross}'

# EMA     -- Amount: -10000, 5 Ema: 213.48, 20 Ema: 213.83, Last cross: Bearish
# MACD    -- Amount: -10000, Macd: -0.5524487505777529, signal: -0.6956048572086206, Last cross: Bearish
# RSI     -- Amount: 0, RSI: 50.435380342685505, Last signal: None
# CLIENTS -- Market maker accepted calls: 5/5, Market taker accepted calls: 0/0
# COVER   -- Amount: -32000.0, To cover: 0
