from .base import Base

# Direction constants
BID, ASK = (0, 1)


class Cover(Base):
    """
    Implements a trading strategy that automatically adjusts its positions to ensure they do not exceed
    a specified position limit. This strategy utilizes the TraderionClient API for interacting with the
    trading platform, allowing it to make informed trading decisions based on the current position and market conditions.

    This class inherits from the :class:`~.Base` class, extending it with specific logic to manage
    trading positions within defined limits.

    :param api: The API interface for interacting with the trading system.
    :type api: :class:`traderion.client.TraderionClient`
    :param options: Strategy-specific options, must include ``position_limit``.
    :type options: dict
    :raises Exception: If ``position_limit`` is not included in ``options`` or does not meet the required criteria.
    """

    def __init__(self, api, options):
        """
        Initializes the Cover strategy with the specified API interface and options.
        """
        super().__init__(api, options)

        if 'position_limit' not in options:
            raise Exception(
                f"Cover options dict must contain a key named position_limit with an integer value")

        position_limit = options['position_limit']

        if type(position_limit) != int:
            raise Exception(f"{position_limit} is not an integer")
        if position_limit < self.api.min_ticket:
            raise Exception(f"{position_limit} is not above min ticket")
        self.position_limit = position_limit
        self.current_amount = self.api.position['amount']

    def on_position_change(self, new_position):
        """
        Callback method to handle position changes. Updates the current amount based on the new position.

        :param new_position: The new position data received from the trading system.
        :type new_position: dict
        """
        self.current_amount = new_position['amount'] / \
            self.api.ticket_unit  # short amount

    def run(self):
        """
        Contains the main logic of the strategy. This method checks if the current position exceeds the position
        limit and attempts to adjust the position to align with the defined limit.
        """
        direction = BID if self.current_amount > 0 else ASK

        remaining_amount = self.get_cover_amount()

        if remaining_amount <= 0:
            return
        else:
            depth = self.api.get_eb_depth()
            ticket = self.api.max_ticket if remaining_amount >= self.api.max_ticket else remaining_amount
            amount = (self.api.hit_price(direction, ticket,
                      depth[direction][0]['price']))['amount']
            remaining_amount -= amount

    def log_info(self):
        """
        Generates and returns a log message describing the current state of the strategy,
        including the current position amount and the amount to be covered.

        :return: A log message summarizing the current state of the strategy.
        :rtype: str
        """
        return f'COVER   -- Amount: {self.current_amount}, To cover: {self.get_cover_amount()}'

    def get_cover_amount(self):
        """
        Calculates the amount needed to adjust the current position to meet the position limit.
        
        :return: The amount that needs to be covered to adhere to the position limit.
        :rtype: int
        """
        return max(abs(self.current_amount) - abs(self.position_limit), 0)
