import threading
import time
from queue import Queue

from .client import TraderionClient


class TraderionBot(object):
    """
        Base class to create a custom trading bot.

        On instantiation it will create a :class:`traderion.client.TraderionClient` with the provided credentials
        and will store it in a member called *api*. Through this member you will call the API methods.

        :param str username: traderion username
        :param str password: traderion password
        :param int room_id: id of the current playing room
        :param float loop_sleep: the frequency (in seconds) of the main loop, defaults to 1
    """

    def __init__(self, username, password, room_id, loop_sleep=1):
        self.loop_sleep = loop_sleep
        self.queue = Queue()
        self.api = TraderionClient(username, password, room_id)
        self.api.register_bot(self)

    def add_event(self, event, payload):
        self.queue.put({'event': event, 'payload': payload})

    def poll_events(self):
        while not self.api.is_finished:
            item = self.queue.get()
            event = item['event']
            payload = item['payload']

            function_map = {
                'on_market_price_change': self.on_market_price_change,
                'on_price_curve_change': self.on_price_curve_change,
                'on_new_client_call': self.on_new_client_call,
                'on_client_deal': self.on_client_deal,
                'on_client_reject': self.on_client_reject,
                'on_eb_depth_change': self.on_eb_depth_change,
                'on_orders_change': self.on_orders_change,
                'on_position_change': self.on_position_change,
                'on_room_status_change': self.on_room_status_change,
            }

            callback = function_map[event]
            if type(payload) is tuple:
                old_state, new_state = payload
                callback(old_state, new_state)
            else:
                callback(payload)

            self.queue.task_done()

    def on_market_price_change(self, old_prices, new_prices):
        """
        Implement this callback to receive notifications when the market prices change.

        See :meth:`traderion.client.TraderionClient.get_market_prices` to see the object's structure.

        :param old_prices: old state of the prices
        :param new_prices: new state of the prices
        :return:
        """
        pass

    def on_price_curve_change(self, old_price_curve, new_price_curve):
        """
        Implement this callback to receive notifications when the price curve changes.

        See :meth:`traderion.client.TraderionClient.get_price_curve` to see the object's structure.

        :param old_price_curve: old state of the price curve
        :param new_price_curve: new state of the price curve
        """
        pass

    def on_new_client_call(self, client_call):
        """
        Implement this callback to receive notifications when a new client call is available.

        Note: if the *client_price* attribute in the client_call is None, then the client needs a quote,
        otherwise you have to either accept or decline the call.

        See :meth:`traderion.client.TraderionClient.get_client_calls` to see the object's structure.

        :param client_call: the new client call
        """
        pass

    def on_client_deal(self, client_call):
        """
        Implement this callback to receive notifications when a client deal is done.

        See :meth:`traderion.client.TraderionClient.get_client_calls` to see the object's structure.

        :param client_call: the accepted client call
        """
        pass

    def on_client_reject(self, client_call):
        """
        Implement this callback to receive notifications when a client deal is rejected.

        See :meth:`traderion.client.TraderionClient.get_client_calls` to see the object's structure.

        :param client_call: the declined client call
        """
        pass

    def on_eb_depth_change(self, old_depth, new_depth):
        """
        Implement this callback to receive notifications when the eb depth changes.

        See :meth:`traderion.client.TraderionClient.get_eb_depth` to see the object's structure.

        :param old_depth: old state of the eb depth
        :param new_depth: new state of the eb depth
        """
        pass

    def on_orders_change(self, old_orders, new_orders):
        """
        Implement this callback to receive notifications when the eb orders change.

        See :meth:`traderion.client.TraderionClient.get_orders` to see the object's structure.

        :param old_orders: old state of the eb orders
        :param new_orders: new state of the eb orders
        """
        pass

    def on_position_change(self, old_position, new_position):
        """
        Implement this callback to receive notifications when position changes.

        See :meth:`traderion.client.TraderionClient.get_position` to see the object's structure.

        :param old_position: old state of the position
        :param new_position: new state of the position
        """
        pass

    def on_room_status_change(self, old_status, new_status):
        """
        Implement this callback to receive notifications when the room status changes.

        The possible values are: 'playing', 'paused', 'finished'.

        :param old_status: old room status
        :param new_status: new room status
        """
        pass

    def log_info(self):
        """
        Implement this method if you want to log something to the console.

        The method will run in a separate thread, so you can use an infinite loop here.

        Check the example for more details.
        """
        pass

    def main_loop(self):
        """
        Implement this method to add your custom bot logic.

        This method will run every *loop_sleep* seconds **while** the room is playing.

        Check the example for more details.
        """
        pass

    def run(self):
        """
        You should not override this method.

        This method calls *main_loop* every *loop_sleep* seconds **while** the room is playing.

        Call this method in your main function to start the bot. Check the example for more details.
        """
        t1 = threading.Thread(target=self.api.poll_for_updates)
        t2 = threading.Thread(target=self.poll_events)
        t3 = threading.Thread(target=self.log_info)

        t1.daemon = True
        t2.daemon = True
        t3.daemon = True

        t1.start()
        t2.start()
        t3.start()

        while not self.api.is_finished:
            if self.api.is_playing:
                self.main_loop()
            time.sleep(self.loop_sleep)

        t1.join()
        t2.join()
        t3.join()
