import random
import time
import sys
import traceback
from traderion.bot import TraderionBot
from traderion.dashboard import Dashboard, format_number
import curses

# Direction constants
BID, ASK = (0, 1)

class ClientsManagerBot(TraderionBot):
    """
    The bot will have a simple strategy:

    Every three seconds it will hit a random price in electronic broker.

    Every five seconds it will add a random order in electronic broker.

    For each client call that needs a quotation it will quote a price
    that has a random spread between 1 and 10 from the market price.

    For each client call that has a price it will accept it only if it can make an arbitrage.
    """

    def __init__(self, username, password, room_id, loop_sleep=1, dashboard=None):
        super().__init__(username, password, room_id, loop_sleep)
        # The timer will count how many times the loop is called
        self.timer = 0
        
        # Dashboard integration
        self.dashboard = dashboard
        
        self.clients_data = {
            'accepted_mm_calls': 0,
            'total_mm_calls': 0,
            'accepted_mt_calls': 0,
            'total_mt_calls': 0,
            'last_client_call': 'None',
            'last_deal_amount': 0.0000,
            'last_deal_price': 0.0000
        }
        if self.dashboard:
            self.dashboard.update_core_data('status', 'Connecting')
            self.dashboard.add_custom_panel(self.get_dashboard_panel())
            self.dashboard.update_core_data('position', {'amount': 0, 'pnl': 0, 'rate': 0})
            self.dashboard.update_core_data('market_prices', {'bid': 0.0000, 'ask': 0.0000, 'open': 0.0000})

    def get_dashboard_panel(self):
        def clients_panel(stdscr, y, x, height, width):
            stdscr.addstr(y, x, "Clients Strategy", curses.A_BOLD)
            stdscr.addstr(y + 1, x + 2, f"MM Calls: {self.clients_data.get('accepted_mm_calls', 0)}/{self.clients_data.get('total_mm_calls', 0)}")
            stdscr.addstr(y + 2, x + 2, f"MT Calls: {self.clients_data.get('accepted_mt_calls', 0)}/{self.clients_data.get('total_mt_calls', 0)}")
            stdscr.addstr(y + 3, x + 2, f"Last Client: {self.clients_data.get('last_client_call', 'N/A')}")
            stdscr.addstr(y + 4, x + 2, f"Last Deal Amt: {format_number(self.clients_data.get('last_deal_amount', 0))}")
            stdscr.addstr(y + 5, x + 2, f"Last Deal Prc: {format_number(self.clients_data.get('last_deal_price', 0))}")
        return clients_panel

    def on_market_price_change(self, old_prices, new_prices):
        """
        Callback for market price changes.
        """
        if self.dashboard:
            self.dashboard.update_core_data('market_prices', new_prices)
            self.dashboard.add_market_event('Market Price Change', f"From {old_prices} to {new_prices}")

    def on_price_curve_change(self, old_price_curve, new_price_curve):
        """
        Callback for price curve changes.
        """
        if self.dashboard:
            self.dashboard.add_market_event('Price Curve Change', "Price curve updated")

    def on_new_client_call(self, client_call):
        """
        Callback for new client calls.
        """
        if self.dashboard:
            self.dashboard.add_bot_event('New Client Call', f"Direction: {'BID' if client_call['direction'] == BID else 'ASK'}")
            self.clients_data['last_client_call'] = f"Direction: {'BID' if client_call['direction'] == BID else 'ASK'}"
            
        if client_call['client_price'] is not None:
            # The client had already set the price, we have to either accept or decline
            self.check_for_arbitrage(client_call)
        else:
            # We need to quote to the client
            self.quote_to_client_call(client_call)

    def on_client_deal(self, client_call):
        """
        Callback for client deals.
        """
        if self.dashboard:
            self.dashboard.add_bot_event('Client Deal', f"Direction: {'BID' if client_call['direction'] == BID else 'ASK'}")
            self.clients_data['accepted_mm_calls'] += 1
            self.clients_data['total_mm_calls'] += 1
            self.clients_data['last_client_call'] = f"Deal: {'BID' if client_call['direction'] == BID else 'ASK'}"
            self.clients_data['last_deal_amount'] = client_call.get('amount', 0.0000)
            self.clients_data['last_deal_price'] = client_call.get('client_price', 0.0000)

    def on_client_reject(self, client_call):
        """
        Callback for client rejections.
        """
        if self.dashboard:
            self.dashboard.add_bot_event('Client Reject', f"Direction: {'BID' if client_call['direction'] == BID else 'ASK'}")
            self.clients_data['total_mm_calls'] += 1
            self.clients_data['last_client_call'] = f"Reject: {'BID' if client_call['direction'] == BID else 'ASK'}"

    def on_eb_depth_change(self, old_depth, new_depth):
        """
        Callback for electronic broker depth changes.
        """
        if self.dashboard:
            self.dashboard.add_market_event('EB Depth Change', "Depth updated")

    def on_orders_change(self, old_orders, new_orders):
        """
        Callback for orders changes.
        """
        if self.dashboard:
            self.dashboard.add_bot_event('Orders Change', f"{len(new_orders)} active orders")

    def on_position_change(self, old_position, new_position):
        """
        Callback for position changes.
        """
        if self.dashboard:
            self.dashboard.update_core_data('position', {
                'amount': new_position['amount'],
                'pnl': new_position['pnl'],
                'rate': new_position['rate']
            })
            self.dashboard.add_performance_metric('Position Change', new_position['pnl'])

    def on_room_status_change(self, old_status, new_status):
        """ Callback for room status updates """
        if self.dashboard:
            status_text = new_status
            self.dashboard.update_core_data('status', status_text)
            if old_status != status_text:
                self.dashboard.add_bot_event('Room Status Change', f"From {old_status} to {status_text}")

    def main_loop(self):
        """
        Main loop of the bot.
        """
        self.timer += self.loop_sleep

        # Update position
        position = self.api.get_position()
        if self.dashboard:
            self.dashboard.update_core_data('position', {
                'amount': position['amount'],
                'pnl': position['pnl'],
                'rate': position.get('rate', 0)
            })

        # Perform actions based on the timer
        if self.timer % 3 == 0:
            self.random_hit_eb()

        if self.timer % 5 == 0:
            self.random_add_eb()

    def get_random_amount(self):
        """
        Returns a random short amount between the min_ticket and the max_ticket.
        """
        room_params = self.api.get_room_parameters()
        min_ticket = room_params['min_ticket']
        max_ticket = room_params['max_ticket']
        return random.randint(min_ticket, max_ticket)

    def get_random_direction(self):
        """
        Returns a random direction (BID or ASK).
        """
        return random.choice([BID, ASK])

    def get_random_spread(self):
        """
        Returns a random spread between 1 and 10 points.
        """
        return random.randint(1, 10)

    def compute_price_from_spread(self, anchor_price, spread, direction):
        """
        Computes a price from the given anchor price, spread, and direction.
        """
        room_params = self.api.get_room_parameters()
        # Compute the value of a pip (FX), cent (EQ), basis point (FI).
        point = 10 ** -(room_params['price_decimals'])
        sgn = (-1, 1)[direction == BID]
        return anchor_price - sgn * spread * point

    def random_hit_eb(self):
        """
        Hits a random price in the electronic broker.
        """
        direction = self.get_random_direction()
        amount = self.get_random_amount()
        eb_depth = self.api.get_eb_depth()
        price = eb_depth[direction][0]['price']  # Best price for the direction

        if self.dashboard:
            self.dashboard.update_core_data('last_action', f"Hitting {'BID' if direction == BID else 'ASK'}")
            
        try:
            self.api.hit_price(direction, amount, price)
            if self.dashboard:
                self.dashboard.add_bot_event('Hit Success', f"{'BID' if direction == BID else 'ASK'} for {amount} at {price}")
                self.dashboard.update_core_data('last_action', f"Hit successful: {'BID' if direction == BID else 'ASK'}")
        except Exception as e:
            if self.dashboard:
                self.dashboard.add_bot_event('Hit Failure', str(e))
                self.dashboard.update_core_data('last_action', f"Hit failed: {e}")

    def random_add_eb(self):
        """
        Adds a random order in the electronic broker.
        """
        direction = self.get_random_direction()
        amount = self.get_random_amount()
        spread = self.get_random_spread()
        market_prices = self.api.get_market_prices()
        anchor_price = (market_prices['bid'], market_prices['ask'])[direction == ASK]
        price = self.compute_price_from_spread(anchor_price, spread, direction)

        if self.dashboard:
            self.dashboard.update_core_data('last_action', f"Adding {'BID' if direction == BID else 'ASK'} order")
            
        try:
            self.api.add_order(direction, amount, price)
            if self.dashboard:
                self.dashboard.add_bot_event('Order Added', f"{'BID' if direction == BID else 'ASK'} for {amount} at {price}")
                self.dashboard.update_core_data('last_action', f"Order added: {'BID' if direction == BID else 'ASK'}")
        except Exception as e:
            if self.dashboard:
                self.dashboard.add_bot_event('Order Failure', str(e))
                self.dashboard.update_core_data('last_action', f"Order failed: {e}")

    def check_for_arbitrage(self, client_call):
        """
        Checks for arbitrage opportunities.
        """
        direction = client_call['direction']
        client_price = client_call['client_price']
        market_prices = self.api.get_market_prices()
        # The anchor price will be the market price on the opposite direction
        anchor_price = (market_prices['bid'], market_prices['ask'])[direction == BID]
        sgn = (-1, 1)[direction == BID]

        if self.dashboard:
            self.dashboard.update_core_data('last_action', f"Checking arbitrage for client call: {client_call['id']}")
            
        if sgn * (client_price - anchor_price) > 0:
            self.api.accept_client_call(client_call['id'])
            if self.dashboard:
                self.dashboard.add_bot_event('Arbitrage Accept', f"Client call {client_call['id']}")
                self.dashboard.update_core_data('last_action', f"Accepted client call: {client_call['id']}")
                self.clients_data['accepted_mt_calls'] += 1
        else:
            self.api.decline_client_call(client_call['id'])
            if self.dashboard:
                self.dashboard.add_bot_event('Arbitrage Decline', f"Client call {client_call['id']}")
                self.dashboard.update_core_data('last_action', f"Declined client call: {client_call['id']}")
        
        if self.dashboard:
            self.clients_data['total_mt_calls'] += 1

    def quote_to_client_call(self, client_call):
        """
        Quotes a price to the client call.
        """
        direction = client_call['direction']
        market_prices = self.api.get_market_prices()
        # The anchor price will be the market price on the opposite direction
        op_direction = (BID, ASK)[direction == BID]
        anchor_price = (market_prices['bid'], market_prices['ask'])[op_direction == ASK]
        spread = self.get_random_spread()
        quote = self.compute_price_from_spread(anchor_price, spread, op_direction)
        
        if self.dashboard:
            self.dashboard.update_core_data('last_action', f"Quoting to client call: {client_call['id']}")
            
        try:
            self.api.quote_client_call(client_call['id'], quote)
            
            # Update dashboard with quote details
            if self.dashboard:
                self.clients_data['last_client_call'] = f"Quote: {'BID' if client_call['direction'] == BID else 'ASK'}"
                self.clients_data['last_deal_price'] = quote
            if self.dashboard:
                self.dashboard.add_bot_event('Quote Success', f"Client call {client_call['id']} at {quote}")
                self.dashboard.update_core_data('last_action', f"Quoted price {quote} to client call: {client_call['id']}")
        except Exception as e:
            if self.dashboard:
                self.dashboard.add_bot_event('Quote Failure', str(e))
                self.dashboard.update_core_data('last_action', f"Quote failed: {e}")

def run_bot(dashboard):
    """
    Run the bot with the dashboard.
    """
    import logging
    import traceback
    import sys
    
    logger = logging.getLogger(__name__)
    
    try:
        # Initialize dashboard with a better status
        dashboard.update_core_data('status', 'Starting')
        dashboard.update_core_data('last_action', 'Initializing bot')
        
        print("Creating bot...")
        # Create the bot
        bot = ClientsManagerBot('vc_trainee1', 'traderion', 19311, dashboard=dashboard)
        
        print("Getting room status...")
        # Log room status
        room_status = bot.api.room_status
        print(f"Room status: {room_status}")
        logger.debug(f"Room status: {room_status}")
        dashboard.add_bot_event('Room Status', f"Current status: {room_status}")
        
        # Update dashboard status
        dashboard.update_core_data('status', f"Room: {room_status}")
        dashboard.update_core_data('last_action', 'Bot ready to trade')
        
        # Check if room is playing
        if not bot.api.is_playing:
            print(f"Room is not playing. Current status: {room_status}")
            dashboard.add_bot_event('Warning', f"Room not playing. Status: {room_status}")
        
        # Add some initial events to show activity
        dashboard.add_bot_event('Bot Initialized', 'Ready to trade')
        
        print("Getting market prices...")
        # Get and log market prices
        try:
            market_prices = bot.api.get_market_prices()
            dashboard.update_core_data('market_prices', market_prices)
            dashboard.add_market_event('Initial Market Prices', f"{market_prices}")
            logger.debug(f"Market prices: {market_prices}")
        except Exception as e:
            print(f"Error getting market prices: {e}")
            print(traceback.format_exc())
            logger.error(f"Error getting market prices: {e}")
            dashboard.add_bot_event('Error', f"Failed to get market prices: {e}")
        
        print("Getting position...")
        # Get and log position
        try:
            position = bot.api.get_position()
            dashboard.update_core_data('position', {
                'amount': position['amount'],
                'pnl': position['pnl'],
                'rate': position.get('rate', 0)
            })
            dashboard.add_bot_event('Initial Position', f"{position}")
            logger.debug(f"Position: {position}")
        except Exception as e:
            print(f"Error getting position: {e}")
            print(traceback.format_exc())
            logger.error(f"Error getting position: {e}")
            dashboard.add_bot_event('Error', f"Failed to get position: {e}")
        
        print("Starting bot...")
        # Run the bot
        dashboard.add_bot_event('Bot Starting', 'Running main loop')
        bot.run()
    except Exception as e:
        print(f"CRITICAL ERROR: {e}")
        print(traceback.format_exc())
        logger.critical(f"Bot initialization failed: {e}")
        logger.critical(traceback.format_exc())
        dashboard.update_core_data('status', f"ERROR: {e}")
        dashboard.update_core_data('last_action', 'Bot failed to start')
        dashboard.add_bot_event('Critical Error', str(e))
        # Sleep to allow dashboard to update before exiting
        time.sleep(5)
        sys.exit(1)

if __name__ == '__main__':
    """
    Entry point for the bot.
    """
    dashboard = Dashboard()
    dashboard.run_dashboard(run_bot, (dashboard,))
