from .base import Base


class TechnicalAnalysis(Base):
    """
    The TechnicalAnalysis class is a base class for implementing technical analysis-based trading strategies.
    It extends the Base class and introduces common functionalities relevant to technical analysis strategies,
    including handling target amount setup and responding to changes in electronic broker depth and price curves.

    Attributes:
        target_amount (int): The target amount for the strategy to act upon.
        depth (list): The current depth of the electronic broker.
        price_curve (list): The current price curve.

    Inherits from:
        Base (class): The base class for all strategies that provides foundational attributes and methods.
    """

    def __init__(self, api, options):
        """
        Initializes the TechnicalAnalysis strategy class with the specified API interface and strategy-specific options.

        Parameters:
            api (object): The API interface for interacting with the trading system.
            options (dict): A dictionary containing strategy-specific options. Must include 'target_amount' key with an
                            integer value indicating the target amount for the strategy.

        Raises:
            Exception: If 'target_amount' key is missing in `options`, is not an integer, or is below the minimum
                       ticket size defined by the API.
        """
        super().__init__(api, options)

        if 'target_amount' not in options:
            raise Exception(f"TechnicalAnalysis options dict must contain a key named target_amount with an integer "
                            f"value")

        target_amount = options['target_amount']

        if type(target_amount) != int:
            raise Exception(f"{target_amount} is not an integer")
        if target_amount < self.api.min_ticket:
            raise Exception(f"{target_amount} is not above min ticket")
        self.target_amount = target_amount

        # Session attributes
        self.depth = self.api.get_eb_depth()
        self.price_curve = [price for (
            time, price) in self.api.get_price_curve()]

    def on_eb_depth_change(self, depth):
        """
        Updates the strategy's view of the electronic broker depth when it changes.

        Parameters:
            new_depth (list): The new electronic broker depth.
        """
        self.depth = depth

    def on_price_curve_change(self, price_curve):
        """
        Updates the strategy's view of the price curve when it changes.

        Parameters:
            new_price_curve (list): The new price curve.
        """
        self.price_curve = price_curve

    def run(self):
        """
        Contains the main logic to be executed by the strategy. This method must be implemented by subclasses.

        Raises:
            NotImplementedError: Indicates that subclasses must implement this method.
        """
        raise NotImplementedError('Strategies must implement the run method')

    def log_info(self):
        """
        Should be implemented by subclasses to log strategy-specific information.

        Raises:
            NotImplementedError: Indicates that subclasses must implement this method.
        """
        raise NotImplementedError(
            'Strategies must implement the log_info method')
