import types

from .base import Base

# Direction constants
BID, ASK = (0, 1)


class Clients(Base):
    """
    This strategy involves responding to client calls by either either quoting a new price
    or deciding to accept or decline the client price, and hitting the EB in search of arbitrage opportunities.

    This strategy uses callable parameters for dynamic spread calculation, allowing for flexibility in how
    spreads are determined in response to market conditions.

    :param api: The API interface for interacting with the trading system.
    :type api: :class:`traderion.client.TraderionClient`
    :param options: Strategy-specific options, must include callable functions for `client_quote_spread` and 
                    `own_quote_spread`.
    :type options: dict
    :raises Exception: If required keys are missing in `options` or if the provided spread parameters are not callable.
    """

    def __init__(self, api, options):
        """
        Initializes the Clients strategy with specified API interface and options, including callable functions for
        determining spreads.
        """
        super().__init__(api, options)

        if not {'client_quote_spread', 'own_quote_spread'} <= set(options):
            raise Exception(f'ClientsStrategy options should contain the following keys: client_quote_spread, '
                            f'own_quote_spread')

        self.get_client_quote_spread = options['client_quote_spread']
        self.get_own_quote_spread = options['own_quote_spread']

        if not all(isinstance(x, types.FunctionType) for x in [self.get_client_quote_spread, self.get_own_quote_spread]):
            raise Exception(
                'Parameters client_quote_spread, own_quote_spread must be callable ')

        self.accepted_mt_calls = 0
        self.total_mt_calls = 0
        self.accepted_mm_calls = 0
        self.total_mm_calls = 0

    def on_new_client_call(self, client_call):
        """
        Responds to new client calls by either quoting a price to the client or deciding to accept or decline
        the client's proposed price.

        :param client_call: The client call to respond to.
        :type client_call: dict
        """
        if client_call['client_price'] is not None:
            # The client had already set the price, we have to either accept or decline
            can_accept = self.check_for_arbitrage(client_call)
            if can_accept:
                self.api.accept_client_call(client_call['id'])
                self.accepted_mt_calls += 1
            else:
                self.api.decline_client_call(client_call['id'])
            self.total_mt_calls += 1
        else:
            # We need to quote to the client
            self.quote_to_client_call(client_call)

    def check_for_arbitrage(self, client_call):
        """
        Determines if accepting a client's proposed price would result in an arbitrage opportunity.

        :param client_call: The client call containing the client's proposed price and direction.
        :type client_call: dict
        :return: True if accepting the call results in arbitrage, False otherwise.
        :rtype: bool
        """
        direction = client_call['direction']
        client_price = client_call['client_price']
        market_prices = self.api.get_market_prices()
        anchor_price = (market_prices['bid'], market_prices['ask'])[
            direction == BID]
        sgn = (-1, 1)[direction == BID]
        point = 10 ** -self.api.price_decimals

        return sgn * (client_price - anchor_price) >= self.get_client_quote_spread() * point

    def quote_to_client_call(self, client_call):
        """
        Quotes a new price to the client based on the strategy's spread calculation.

        :param client_call: The client call requiring a quote.
        :type client_call: dict
        """
        direction = client_call['direction']  # BID
        market_prices = self.api.get_market_prices()
        op_direction = (BID, ASK)[direction == BID]
        anchor_price = (market_prices['bid'], market_prices['ask'])[
            op_direction == ASK]  # market_prices['ask']

        spread = self.get_own_quote_spread()
        quote = self.compute_price_from_spread(
            anchor_price, spread, op_direction)
        self.api.quote_client_call(client_call['id'], quote)

    def compute_price_from_spread(self, anchor_price, spread, direction):
        """
        Computes a quote price from a given spread and direction relative to an anchor price.

        :param anchor_price: The market price serving as an anchor for the quote calculation.
        :type anchor_price: float
        :param spread: The spread to apply to the anchor price.
        :type spread: float
        :param direction: The direction of the quote (BID or ASK).
        :type direction: int
        :return: The computed quote price.
        :rtype: float
        """
        # Usually, the anchor_price will be the market bid, or the market ask.
        # If the direction is BID, then it will return a price smaller than the market bid.
        # If the direction is ASK, then it will return a price greater than the market ask.

        # Compute the value of a pip (FX), cent (EQ), basis point (FI).
        point = 10 ** -self.api.price_decimals
        sgn = (-1, 1)[direction == BID]
        return anchor_price - sgn * spread * point

    def on_client_deal(self, client_call):
        """
        Callback method for when a client deal is accepted.

        :param client_call: The client call that was accepted.
        :type client_call: dict
        """
        self.accepted_mm_calls += 1
        self.total_mm_calls += 1

    def on_client_reject(self, client_call):
        """
        Callback method for when a client call is rejected.

        :param client_call: The client call that was rejected.
        :type client_call: dict
        """
        self.total_mm_calls += 1

    def log_info(self):
        """
        Generates and returns a log message summarizing the strategy's activity, including the number of accepted
        market maker and market taker calls.

        :return: A log message summarizing the strategy's activity.
        :rtype: str
        """
        return f'CLIENTS -- Market maker accepted calls: {self.accepted_mm_calls}/{self.total_mm_calls}, Market taker ' \
               f'accepted calls: {self.accepted_mt_calls}/{self.total_mt_calls}'
