class Base:
    """
    The Base class serves as the foundation for all strategy implementations. 
    It defines a set of methods that can be overridden by derived classes to 
    react to different events in the trading environment.

    Methods:
        on_position_change(new_position): Called when the bot position changes.
        on_price_curve_change(new_price_curve): Called when the price curve changes.
        on_eb_depth_change(new_depth): Called when the electronic broker depth changes.
        on_new_client_call(client_call): Called when a new client call appears.
        on_client_deal(client_call): Called when a client deal occurs.
        on_client_reject(client_call): Called when a client call is rejected.
        run(): Contains the main logic to be executed by the strategy.
        log_info(): Should be implemented by subclasses to log strategy-specific information.
    """

    def __init__(self, api, options):
        """
        Initializes the Base strategy class with the provided API interface and strategy-specific options.

        Parameters:
            api (object): The API interface for interacting with the trading system.
            options (dict): A dictionary of strategy-specific options.
        
        Raises:
            Exception: If `options` is not a dictionary.
        """
        if not isinstance(options, dict):
            raise Exception(f"Options must be a dict")

        self.api = api

    def on_position_change(self, new_position):
        """Called when the bot position changes. Override in subclass."""
        pass

    def on_price_curve_change(self, new_price_curve):
        """Called when the price curve changes. Override in subclass."""
        pass

    def on_eb_depth_change(self, new_depth):
        """Called when the electronic broker depth changes. Override in subclass."""
        pass

    def on_new_client_call(self, client_call):
        """Called when a new client call appears. Override in subclass."""
        pass

    def on_client_deal(self, client_call):
        """Called when a client deal occurs. Override in subclass."""
        pass

    def on_client_reject(self, client_call):
        """Called when a client call is rejected. Override in subclass."""
        pass

    def run(self):
        """Contains the main logic to be executed by the strategy. Override in subclass."""
        pass

    def log_info(self):
        """
        Should be implemented by subclasses to log strategy-specific information.
        
        Raises:
            NotImplementedError: If the subclass does not implement this method.
        """
        raise NotImplementedError('Strategies must implement the log_info method')
