import curses
import threading
import time
import collections
import logging
import traceback
import os
import sys
from datetime import datetime

# Configure logging
log_dir = os.path.join('.', 'logs')
os.makedirs(log_dir, exist_ok=True)
logging.basicConfig(
    level=logging.DEBUG,
    format='%(asctime)s - %(levelname)s - %(message)s',
    filename=os.path.join(log_dir, 'dashboard_debug.log'),
    filemode='w'
)
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

def format_number(val):
    """
    Format a numeric value with thousands separators and up to 4 decimals.
    If the value is an integer, it is displayed without decimals.
    """
    try:
        f = float(val)
        if f.is_integer():
            return f"{int(f):,}"
        else:
            return f"{f:,.4f}"
    except (TypeError, ValueError):
        return str(val)

class DashboardError(Exception):
    """Custom exception for dashboard-related errors."""
    pass

class Dashboard:
    """
    A centralized dashboard system for Traderion bots.
    """
    def __init__(self, title="Traderion Bot Dashboard", max_history=100):
        logger.debug(f"Initializing Dashboard with title: {title}")
        self.title = title
        self.data_lock = threading.Lock()
        self.max_history = max_history
        
        self.core_data = {
            'position': {'amount': 0.0000, 'pnl': 0.0000, 'rate': 0.0000},
            'market_prices': {'bid': 0.0000, 'ask': 0.0000, 'open': 0.0000},
            'status': 'Initializing',
            'last_action': 'Starting bot',
        }
        
        self.market_history = collections.deque(maxlen=max_history)
        self.event_history = collections.deque(maxlen=max_history)
        self.performance_history = collections.deque(maxlen=max_history)
        
        self.custom_panels = []
        
        self.add_bot_event('Dashboard', 'Initialized')
        logger.debug("Dashboard initialization complete")
        
    def add_custom_panel(self, panel_function):
        logger.debug(f"Adding custom panel: {panel_function.__name__ if hasattr(panel_function, '__name__') else 'anonymous'}")
        self.custom_panels.append(panel_function)
    
    def _round_value(self, value):
        if value is None:
            return 0.0000
        try:
            return round(float(value), 4)
        except (TypeError, ValueError):
            return 0.0000
    
    def _safe_get(self, dictionary, key, default=0.0000):
        return dictionary.get(key, default) if dictionary is not None else default
        
    def update_core_data(self, key, value):
        logger.debug(f"Updating core data: {key} = {value}")
        with self.data_lock:
            if key in self.core_data:
                if isinstance(self.core_data[key], dict):
                    self.core_data[key] = {k: self._round_value(v) for k, v in value.items()}
                elif key == 'status':
                    self.core_data[key] = str(value)
                else:
                    self.core_data[key] = self._round_value(value)
            else:
                self.core_data[key] = value

    def add_market_event(self, event_type, data):
        logger.debug(f"Adding market event: {event_type} - {data}")
        with self.data_lock:
            self.market_history.append({
                'timestamp': datetime.now(),
                'type': event_type,
                'data': data
            })
            
    def add_bot_event(self, event_type, data):
        logger.debug(f"Adding bot event: {event_type} - {data}")
        with self.data_lock:
            self.event_history.append({
                'timestamp': datetime.now(),
                'type': event_type,
                'data': data
            })
            
    def add_performance_metric(self, metric_type, value):
        logger.debug(f"Adding performance metric: {metric_type} - {value}")
        with self.data_lock:
            self.performance_history.append({
                'timestamp': datetime.now(),
                'type': metric_type,
                'value': self._round_value(value)
            })
                
    def draw_dashboard(self, stdscr):
        try:
            stdscr.clear()
            height, width = stdscr.getmaxyx()
            title_start = (width - len(self.title)) // 2
            stdscr.addstr(0, title_start, self.title, curses.A_BOLD | curses.A_UNDERLINE)
            stdscr.box()
            stdscr.hline(2, 1, curses.ACS_HLINE, width - 2)

            with self.data_lock:
                status = self.core_data.get('status', 'Unknown')
                stdscr.addstr(1, 2, f"Room Status: {status}", curses.A_BOLD)
                if status == 'Connecting':
                    stdscr.addstr(height // 2, 2, "Connecting to server...", curses.A_DIM)
                else:
                    position = self.core_data.get('position', {})
                    amount = self._safe_get(position, 'amount')
                    pnl = self._safe_get(position, 'pnl')
                    rate = self._safe_get(position, 'rate')
                    stdscr.addstr(3, 2, "Position Overview", curses.A_BOLD)
                    stdscr.addstr(4, 4, f"Amount:  {format_number(amount):>15}")
                    stdscr.addstr(5, 4, f"PnL:     {format_number(pnl):>15}")
                    stdscr.addstr(6, 4, f"Rate:    {rate:>15.4f}")

                    market_prices = self.core_data.get('market_prices', {})
                    bid = self._safe_get(market_prices, 'bid')
                    ask = self._safe_get(market_prices, 'ask')
                    open_price = self._safe_get(market_prices, 'open')
                    stdscr.addstr(8, 2, "Market Prices", curses.A_BOLD)
                    stdscr.addstr(9, 4, f"Bid:     {format_number(bid):>15}")
                    stdscr.addstr(10, 4, f"Ask:     {format_number(ask):>15}")
                    stdscr.addstr(11, 4, f"Open:    {format_number(open_price):>15}")

                    last_action = str(self.core_data.get('last_action', 'No action'))
                    stdscr.addstr(13, 2, "Bot Status", curses.A_BOLD)
                    stdscr.addstr(14, 4, f"Action:  {last_action}")

                    custom_panel_y = 16
                    for panel in self.custom_panels:
                        panel_height = 7
                        if custom_panel_y + panel_height < height - 3:
                            try:
                                panel(stdscr, custom_panel_y, 2, panel_height, width - 4)
                                custom_panel_y += panel_height + 1
                            except Exception as panel_error:
                                logger.error(f"Error drawing panel: {panel_error}")

                    events_panel_y = custom_panel_y
                    stdscr.addstr(events_panel_y, 2, "Recent Events", curses.A_BOLD)
                    for i, event in enumerate(reversed(list(self.event_history))):
                        if events_panel_y + 1 + i < height - 2:
                            try:
                                event_time = event.get('timestamp', datetime.now()).strftime('%H:%M:%S')
                                event_details = str(event.get('data', 'No details'))
                                event_type = str(event.get('type', 'Unknown'))
                                event_line = f"{event_time} | {event_type}: {event_details}"
                                stdscr.addstr(events_panel_y + 1 + i, 4, event_line[:width-6])
                            except Exception as event_error:
                                logger.error(f"Error drawing event: {event_error}")

            stdscr.addstr(height - 2, 2, "Press 'q' to quit")
            stdscr.refresh()

        except Exception as e:
            logger.error(f"Dashboard drawing error: {e}")
            logger.error(traceback.format_exc())
            try:
                stdscr.clear()
                stdscr.addstr(0, 0, f"Dashboard Error: {e}")
                stdscr.addstr(1, 0, "Check ./logs/dashboard_debug.log for details")
                stdscr.refresh()
                time.sleep(3)
            except:
                pass

    def run_dashboard(self, bot_thread_target, bot_thread_args=()):
        def dashboard_loop(stdscr):
            try:
                curses.curs_set(0)
                stdscr.timeout(100)
            except Exception as e:
                logger.error(f"Curses initialization error: {e}")
                print(f"Curses initialization error: {e}")
                print(traceback.format_exc())
            
            bot_thread = threading.Thread(target=self._run_bot_with_error_handling, 
                                        args=(bot_thread_target, bot_thread_args, stdscr))
            bot_thread.daemon = True
            
            try:
                logger.debug("Starting bot thread")
                print("Starting bot thread...")
                bot_thread.start()
                logger.debug("Bot thread started successfully")
                print("Bot thread started successfully")
            except Exception as e:
                logger.error(f"Bot thread start error: {e}")
                logger.error(traceback.format_exc())
                print(f"Bot thread start error: {e}")
                print(traceback.format_exc())
                try:
                    stdscr.clear()
                    stdscr.addstr(0, 0, f"Bot Thread Error: {e}")
                    stdscr.addstr(1, 0, "Check ./logs/dashboard_debug.log for details")
                    stdscr.refresh()
                    time.sleep(3)
                except:
                    pass
                return
            
            while True:
                try:
                    self.draw_dashboard(stdscr)
                    key = stdscr.getch()
                    if key == ord('q'):
                        break
                    time.sleep(0.1)
                except Exception as e:
                    logger.error(f"Dashboard loop error: {e}")
                    logger.error(traceback.format_exc())
                    print(f"Dashboard loop error: {e}")
                    print(traceback.format_exc())
                    try:
                        stdscr.clear()
                        stdscr.addstr(0, 0, f"Dashboard Error: {e}")
                        stdscr.addstr(1, 0, "Check ./logs/dashboard_debug.log for details")
                        stdscr.refresh()
                        time.sleep(3)
                    except:
                        pass
                    break
        
        try:
            logger.debug("Starting dashboard")
            print("Starting dashboard...")
            curses.wrapper(dashboard_loop)
        except KeyboardInterrupt:
            logger.info("Bot stopped by user.")
            print("Bot stopped by user.")
        except Exception as e:
            logger.error(f"Dashboard run error: {e}")
            logger.error(traceback.format_exc())
            print(f"CRITICAL ERROR: {e}")
            print(traceback.format_exc())
            print("Check ./logs/dashboard_debug.log for details")
            sys.exit(1)
            
    def _run_bot_with_error_handling(self, bot_thread_target, bot_thread_args, stdscr):
        try:
            bot_thread_target(*bot_thread_args)
        except Exception as e:
            logger.critical(f"Bot thread error: {e}")
            logger.critical(traceback.format_exc())
            print(f"CRITICAL ERROR in bot thread: {e}")
            print(traceback.format_exc())
            self.update_core_data('status', f"ERROR: {e}")
            self.update_core_data('last_action', 'Bot crashed')
            self.add_bot_event('Critical Error', str(e))
            try:
                stdscr.clear()
                stdscr.addstr(0, 0, f"Bot Error: {e}")
                stdscr.addstr(1, 0, "Check ./logs/dashboard_debug.log for details")
                stdscr.addstr(2, 0, "Press 'q' to quit")
                stdscr.refresh()
            except:
                pass