import time
from datetime import datetime

import pytz
import requests

from .exceptions import LoginError, RoomNotFound, ServerError, ValidationError, TradingError


API_URL = 'https://simulator.traderion.com/'
CLIENT_WAITING, CLIENT_PENDING, CLIENT_NO_DEAL, CLIENT_ACCEPTED = list(range(4))
BID, ASK = (0, 1)


class TraderionClient(object):
    """
    Main API interface.

    On instantiation, the client will authenticate with the provided credentials and will
    fetch the initial room data.

    :param str username: traderion username
    :param str password: traderion password
    :param int room_id: id of the current playing room

    :raises LoginError: if the credentials are not correct
    :raises RoomNotFound: if the room is is not correct
    """

    def __init__(self, username, password, room_id):
        self.room_id = room_id
        self.room_status = None
        self.bot = None
        self.poll_times = {
            'badges': None,
            'blotters': None,
            'cb_prices': None,
            'clients': None,
            'electronic_broker': None,
            'level_up': None,
            'macro': None,
            'positions': None,
            'room': None,
            'statistics': None,
            'trading': None,
        }

        # SESSION INFO
        self.asset_class = None
        self.swift = None
        self.min_ticket = None
        self.max_ticket = None
        self.price_decimals = None
        self.ticket_unit = None

        # TRADING INFO
        self.position = {}
        self.orders = {}
        self.eb_depth = {}
        self.client_calls = {}
        self.market_prices = {}
        self.price_curve = []

        # INIT DATA
        self.session = self.create_session(username, password)
        self.get_init_info()
        self.initial_poll()

    def create_session(self, username, password):
        headers = {
            "X-Internal-Login": "1"  # Custom header to bypass ALB redirect
        }

        response = requests.get(API_URL + 'accounts/login/', headers=headers)
        data = {
            'username': username,
            'password': password,
            'csrfmiddlewaretoken': response.cookies['csrftoken'],
        }

        session = requests.Session()
        session.headers.update({'Referer': API_URL})
        response = session.post(API_URL + 'accounts/login/', data=data, cookies=response.cookies)
        if 'Your username and password didn\'t match. Please try again.' in response.text:
            raise LoginError()
        session.headers.update({'X-CSRFToken': session.cookies['csrftoken']})
        return session

    def get_init_info(self):
        response = self.session.get(API_URL + 'treasury/get_init_info/?r=' + str(self.room_id))
        if response.status_code == 404:
            raise RoomNotFound()

        data = response.json()
        if 'room' not in data:
            print("Room finished")
            exit(0)

        self.room_status = data['room']['status']
        self.asset_class = data['room']['asset_class']
        self.swift = data['room']['underlying_asset']
        ac_name = self.asset_class['name']
        self.min_ticket = data['rules'][ac_name]['min_ticket']
        self.max_ticket = data['rules'][ac_name]['max_ticket']
        self.price_decimals = data['rules'][ac_name]['price_decimals']
        self.ticket_unit = data['rules'][ac_name]['ticket_unit']

    def initial_poll(self):
        payload = self.poll_times
        payload['room_id'] = self.room_id
        response = self.session.post(API_URL + 'pull/', json=payload)
        poll_data = response.json()

        for app, d in poll_data.items():
            data = d['data']
            if app == 'room':
                self.room_status = data['room']['status'].lower()
            elif app == 'positions':
                self.position = data['positions'][0]
            elif app == 'electronic_broker':
                self.orders = {order['id']: order for order in data['banks_prices']}
                self.eb_depth = {
                    BID: data['eb_depth'][str(self.swift['id'])][str(BID)],
                    ASK: data['eb_depth'][str(self.swift['id'])][str(ASK)],
                }
            elif app == 'cb_prices':
                self.price_curve = [
                    (self.parse_datetime(date), float(price))
                    for date, price in data['cb_price_curves'][str(self.swift['id'])]
                ]
                market_price = data['cb_prices'][str(self.swift['id'])]
                self.market_prices = {
                    'open': float(market_price['open_price']),
                    'bid': float(market_price['bid']),
                    'ask': float(market_price['ask']),
                }

    def register_bot(self, bot):
        self.bot = bot

    @property
    def is_playing(self):
        """
        Property

        :return: True if the room is playing, False otherwise
        :rtype: bool
        """
        return self.room_status == 'playing'

    @property
    def is_paused(self):
        """
        Property

        :return: True if the room is paused, False otherwise
        :rtype: bool
        """
        return self.room_status == 'paused'

    @property
    def is_finished(self):
        """
        Property

        :return: True if the room is finished, False otherwise
        :rtype: bool
        """
        return self.room_status == 'finished'

    def parse_datetime(self, datetime_str):
        date, time = datetime_str.split('T')
        time = time[:8]
        return datetime.strptime(date + ' ' + time, '%Y-%m-%d %H:%M:%S').replace(tzinfo=pytz.utc)

    def get_room_parameters(self):
        """
        :return: an object with the room parameters
        :rtype: dict

        Room parameters:
            *asset_class*: the name of the asset class (ex: FX, EQ, FI)

            *swift*: the name of the swift (ex: EUR/USD, TSLA, US10Y)

            *min_ticket*: the minimum dealing ticket (in short amount, ex: 1 for FX)

            *max_ticket*: the maximum dealing ticket (in short amount, ex: 10 for FX)

            *price_decimals*: the number of decimals in a quotation (ex: 4 for FX, 2 for EQ)

            *ticket_unit*: the full amount of a ticket of value 1 (ex: 1000000 for FX, 1 for EQ, 1000000 for FI)
        """
        return {
            'asset_class': self.asset_class['name'],
            'swift': self.swift['name'],
            'min_ticket': self.min_ticket,
            'max_ticket': self.max_ticket,
            'price_decimals': self.price_decimals,
            'ticket_unit': self.ticket_unit,
        }

    def get_market_prices(self):
        """
        :return: an object with the market prices (as float)
        :rtype: dict

        Market prices:
            *bid*: the current market bid

            *ask*: the current market ask

            *open*: the open price of the scenario
        """
        return self.market_prices

    def get_price_curve(self):
        """
        :return: an array of market prices represented as (date, price) tuples
        :rtype: list
        """
        return self.price_curve

    def get_position(self):
        """
        :return: an object with all the info about your current position
        :rtype: dict

        Position properties:
            *amount*: full amount of the position

            *rate*: average rate of the position

            *pnl*: the PnL expressed in the reporting currency of the swift

            *converted_pnl*: the PnL expressed in USD

            *pnl_percentage*: the PnL relative to the position limit

            *return_on_volume*: the PnL relative to the total traded volume

            *limit*: position limit (full amount)

            *mat*: management action trigger (1% of the position limit)

            *risk*: the absolute amount over the position limit

            *limit_utilization*: (percent) position amount from position limit

            *headroom*: the available amount to trade before the position limit is broken
        """
        return self.serialize_position(self.position)

    def get_orders(self):
        """
        :return: an array of electronic broker orders
        :rtype: list

        EB Order:
            *id*: order id

            *amount*: order amount (short amount)

            *direction*: 0/1 (= BID/ASK)

            *price*: order price

            *date*: date added
        """
        return [self.serialize_eb_order(order) for order in self.orders.values()]

    def get_eb_depth(self):
        """
        :return: an object with the depth of the market on both sides (BID/ASK)
        :rtype: dict

        Example: {

        0: [{'amount': 10, 'count': 1, 'price': 4.2340}, ...],

        1: [{'amount': 10, 'count': 1, 'price': 4.2350}, ...]

        }

        Each entry in the eb depth has three attributes:
            *price*: the price of the orders

            *amount*: the total amount of the orders with that price (short amount)

            *count*: the total number of orders with that price
        """
        return self.eb_depth

    def get_client_calls(self):
        """
        :return: an array with active client calls
        :rtype: list

        Client Call:
            *id*: call id

            *client*: name of the client

            *date*: call date

            *amount*: call amount (short amount)

            *direction*: 0/1 (BID/ASK, client buys/client sells)

            *is_hedging*: whether the client is a hedge fund or not

            *is_binding*: whether there is a binding contract with the client or not

            *max_spread*: the maximum spread you can quote to the client (None if is_binding is False)

            *client_price*: the price at which the client wants to buy or sell (None if the client requests quote)

            *trader_quote*: the quote offered by the trader (None if client_price is not None or the trader hasn't quoted yet)
        """
        return [
            self.serialize_client_call(call) for call in self.client_calls.values()
            if call['status'] == CLIENT_WAITING
        ]

    def serialize_position(self, position):
        return {
            'amount': position['amount'],
            'rate': position['rate'] if self.asset_class['name'] != 'FI' else position['yield_'],
            'pnl': position['pnl'],
            'converted_pnl': position['converted_pnl'],
            'pnl_percentage': position['pnl_percentage'],
            'return_on_volume': position['return_on_volume'],
            'limit': position['limit'],
            'mat': position['mat'],
            'risk': position['risk'],
            'limit_utilization': position['limit_utilization'],
            'headroom': position['headroom'],
        }

    def serialize_eb_order(self, order):
        return {
            'id': order['id'],
            'amount': order['amount'],
            'direction': order['direction'],
            'price': float(order['price']),
            'date': self.parse_datetime(order['date'])
        }

    def serialize_client_call(self, call):
        return {
            'id': call['id'],
            'client': call['call_info']['client']['name'],
            'date': self.parse_datetime(call['call_info']['date']),
            'amount': call['call_info']['amount'],
            'direction': call['call_info']['direction'],
            'is_hedging': call['call_info']['is_hedging'],
            'is_binding': call['call_info']['is_binding'],
            'max_spread': call['call_info']['max_spread'],
            'client_price': None if call['call_info']['price'] is None else float(call['call_info']['price']),
            'trader_quote': None if call['price'] is None else float(call['price']),
        }

    def make_api_call(self, url, data):
        data['room_id'] = self.room_id
        response = self.session.post(API_URL + url, json=data)

        try:
            json_response = response.json()
        except:
            raise ServerError()

        if type(json_response) != dict:
            raise ValidationError(json_response)
        elif json_response.get('error') is True:
            raise TradingError(json_response['error_message'])

        return json_response

    def poll_for_updates(self):
        while not self.is_finished:
            try:
                data = self.make_api_call('pull/', self.poll_times)
            except:
                data = None

            poll_data = {}
            if type(data) == dict:
                for app, d in data.items():
                    poll_data[app] = d['data']
                    self.poll_times[app] = d['timestamp']
                self.update_trading_data(poll_data)

            if self.room_status == 'paused':
                time.sleep(5)

    def update_trading_data(self, poll_data):
        function_map = {
            'room': self.update_room,
            'cb_prices': self.update_market_prices,
            'electronic_broker': self.update_electronic_broker,
            'clients': self.update_client_calls,
            'positions': self.update_position,
        }
        for key, function in function_map.items():
            if key in poll_data:
                function(poll_data[key])

    def update_room(self, data):
        old_status = self.room_status
        self.room_status = data['room']['status'].lower()
        self.bot.add_event('on_room_status_change', (old_status, self.room_status))

    def update_client_calls(self, data):
        old_client_calls = self.client_calls
        self.client_calls = {call['id']: call for call in data['enquiries']}
        for call_id, call in self.client_calls.items():
            serialized_call = self.serialize_client_call(call)
            if call_id not in old_client_calls:
                if call['status'] == CLIENT_WAITING:
                    self.bot.add_event('on_new_client_call', serialized_call)
            elif call['status'] != old_client_calls[call_id]['status']:
                if call['status'] == CLIENT_ACCEPTED:
                    self.bot.add_event('on_client_deal', serialized_call)
                elif call['status'] == CLIENT_NO_DEAL:
                    self.bot.add_event('on_client_reject', serialized_call)

    def update_electronic_broker(self, data):
        old_orders = self.orders
        self.orders = {order['id']: order for order in data['banks_prices']}
        if old_orders != self.orders:
            old_serialized = [self.serialize_eb_order(order) for order in old_orders.values()]
            new_serialized = [self.serialize_eb_order(order) for order in self.orders.values()]
            self.bot.add_event('on_orders_change', (old_serialized, new_serialized))

        old_depth = self.eb_depth
        self.eb_depth = {
            BID: data['eb_depth'][str(self.swift['id'])][str(BID)],
            ASK: data['eb_depth'][str(self.swift['id'])][str(ASK)],
        }
        if old_depth != self.eb_depth:
            self.bot.add_event('on_eb_depth_change', (old_depth, self.eb_depth))

    def update_position(self, data):
        old_position = self.position
        self.position = data['positions'][0]
        self.bot.add_event('on_position_change', (old_position, self.position))

    def update_market_prices(self, data):
        old_price_curve = self.price_curve
        self.price_curve = [
            (self.parse_datetime(date), float(price))
            for date, price in data['cb_price_curves'][str(self.swift['id'])]
        ]
        if old_price_curve != self.price_curve:
            self.bot.add_event('on_price_curve_change', (old_price_curve, self.price_curve))

        market_price = data['cb_prices'][str(self.swift['id'])]
        old_prices = self.market_prices
        self.market_prices = {
            'open': float(market_price['open_price']),
            'bid': float(market_price['bid']),
            'ask': float(market_price['ask']),
        }
        if old_prices != self.market_prices:
            self.bot.add_event('on_market_price_change', (old_prices, self.market_prices))

    def hit_price(self, direction, amount, price):
        """
        API call to hit a price in electronic broker.

        :param int direction: 0/1 (BID/ASK) direction to hit
        :param int amount: hit amount (short amount)
        :param float price: the price to hit
        :return: server response, an object with {'amount': the amount that was actually hit (**short amount**)}
        :rtype: dict
        :raises ServerError: if there is an internal server error
        :raises ValidationError: if there is an issue with the payload
        :raises TradingError: if the action cannot be completed because of other trading constraints
        """
        payload = {
            'action': 'hit',
            'direction': direction,
            'amount': amount,
            'price': price,
        }
        return self.make_api_call('treasury/electronic_broker/price_action/', payload)

    def add_order(self, direction, amount, price):
        """
        API call to add an order in electronic broker.

        :param int direction: 0/1 (BID/ASK) direction to add
        :param int amount: order amount (short amount)
        :param float price: order price
        :return: server response (nothing important for this action)
        :rtype: dict
        :raises ServerError: if there is an internal server error
        :raises ValidationError: if there is an issue with the payload
        :raises TradingError: if the action cannot be completed because of other trading constraints
        """
        payload = {
            'action': 'add',
            'direction': direction,
            'amount': amount,
            'price': price,
        }
        return self.make_api_call('treasury/electronic_broker/price_action/', payload)

    def cancel_order(self, order_id):
        """
        API call to cancel an order.

        :param int order_id: order id
        :return: server response (nothing important for this action)
        :rtype: dict
        :raises ServerError: if there is an internal server error
        :raises ValidationError: if there is an issue with the payload
        :raises TradingError: if the action cannot be completed because of other trading constraints
        """
        payload = {
            'action': 'cancel',
            'order_id': order_id,
        }
        return self.make_api_call('treasury/electronic_broker/price_action/', payload)

    def cancel_all_orders(self):
        """
        API call to cancel all own orders.

        :return: server response (nothing important for this action)
        :rtype: dict
        :raises ServerError: if there is an internal server error
        :raises ValidationError: if there is an issue with the payload
        :raises TradingError: if the action cannot be completed because of other trading constraints
        """
        payload = {
            'action': 'cancel',
            'underlying_asset': self.swift['id'],
        }
        return self.make_api_call('treasury/electronic_broker/prices_action/', payload)

    def quote_client_call(self, call_id, quote):
        """
        API call to answer a client call.

        Note: the client call must have *client_price* set to None

        :param int call_id: call id
        :param float quote: the price offered to the client
        :return: server response (nothing important for this action)
        :rtype: dict
        :raises ServerError: if there is an internal server error
        :raises ValidationError: if there is an issue with the payload
        :raises TradingError: if the action cannot be completed because of other trading constraints
        """
        payload = {
            'action': 'answer',
            'enquiry_id': call_id,
            'price': quote,
        }
        return self.make_api_call('treasury/clients/enquiry_action/', payload)

    def accept_client_call(self, call_id):
        """
        API call to accept a client call.

        Note: the client call must not have *client_price* set to None

        :param int call_id: call id
        :return: server response (nothing important for this action)
        :rtype: dict
        :raises ServerError: if there is an internal server error
        :raises ValidationError: if there is an issue with the payload
        :raises TradingError: if the action cannot be completed because of other trading constraints
        """
        payload = {
            'action': 'accept',
            'enquiry_id': call_id,
        }
        return self.make_api_call('treasury/clients/enquiry_action/', payload)

    def decline_client_call(self, call_id):
        """
        API call to decline a client call.

        Note: the client call must not have *client_price* set to None

        :param int call_id: call id
        :return: server response (nothing important for this action)
        :rtype: dict
        :raises ServerError: if there is an internal server error
        :raises ValidationError: if there is an issue with the payload
        :raises TradingError: if the action cannot be completed because of other trading constraints
        """
        payload = {
            'action': 'decline',
            'enquiry_id': call_id,
        }
        return self.make_api_call('treasury/clients/enquiry_action/', payload)
